/*
 * Part.java February 2007
 *
 * Copyright (C) 2007, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.http;

import java.io.IOException;
import java.io.InputStream;

/**
 * The <code>Part</code> object is used to represent a part within
 * a request message. Typically a part represents either a text
 * parameter or a file, with associated headers. The contents of
 * the part can be acquire as an <code>InputStream</code> or as a
 * string encoded in the default HTTP encoding ISO-8859-1 or in
 * the encoding specified with the Content-Type header.
 * 
 * @author Niall Gallagher
 * 
 * @see org.simpleframework.http.Form
 */
public interface Part {
   
   /**
    * This method is used to determine the type of a part. Typically
    * a part is either a text parameter or a file. If this is true
    * then the content represented by the associated part is a file.
    *
    * @return this returns true if the associated part is a file
    */
   public boolean isFile();
   
   /**
    * This method is used to acquire the name of the part. Typically
    * this is used when the part represents a text parameter rather
    * than a file. However, this can also be used with a file part.
    * 
    * @return this returns the name of the associated part
    */
   public String getName();
   
   /**
    * This method is used to acquire the file name of the part. This
    * is used when the part represents a text parameter rather than 
    * a file. However, this can also be used with a file part.
    *
    * @return this returns the file name of the associated part
    */
   public String getFileName();
   
   /**
    * This is used to acquire the header value for the specified 
    * header name. Providing the header values through this method
    * ensures any special processing for a know content type can be
    * handled by an application.
    * 
    * @param name the name of the header to get the value for
    * 
    * @return value of the header mapped to the specified name
    */
   public String getHeader(String name);
   
   /**
    * This is used to acquire the content of the part as a string.
    * The encoding of the string is taken from the content type. 
    * If no content type is sent the content is decoded in the
    * standard default of ISO-8859-1.
    * 
    * @return this returns a string representing the content
    * 
    * @throws IOException thrown if the content can not be created
    */
   public String getContent() throws IOException;

   /**
    * This is used to acquire an <code>InputStream</code> for the
    * part. Acquiring the stream allows the content of the part to
    * be consumed by reading the stream. Each invocation of this
    * method will produce a new stream starting from the first byte.
    * 
    * @return this returns the stream for this part object
    * 
    * @throws IOException thrown if the stream can not be created
    */
	public InputStream getInputStream() throws IOException;
	  
	/**
	 * This is used to acquire the content type for this part. This
	 * is typically the type of content for a file part, as provided
	 * by a MIME type from the HTTP "Content-Type" header.
	 * 
	 * @return this returns the content type for the part object
	 */
	public ContentType getContentType();	
}
