/*
 * Channel.java February 2007
 *
 * Copyright (C) 2007, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.http.core;

import java.nio.channels.SocketChannel;
import java.util.Map;

/**
 * The <code>Channel</code> interface represents a connected channel
 * through which data can be sent and received. Typically a channel
 * will have a connected TCP socket, which can be used to determine
 * when the channel is read ready, and write ready. A channel can
 * also contain a bag of attributes used to describe the connection.
 * <p>
 * Reading and writing to a channel is performed using two special
 * interfaces. The first is the <code>Cursor</code> object which is
 * used to read data from the channel in a non-blocking manner. This
 * can also be used to reset data if it has read too much. To write
 * the <code>Sender</code> can be used, this provides a blocking
 * interface much like a conventional output stream.
 *
 * @author Niall Gallagher
 */ 
interface Channel {
   
   /**
    * This is used to determine if the channel is secure and that
    * data read from and data written to the request is encrypted.
    * Channels transferred over SSL are considered secure and will
    * have this return true, otherwise it will return false.
    * 
    * @return true if this is secure for reading and writing
    */
   public boolean isSecure();
   
   /**
    * This is the connected socket channel associated with this. In
    * order to determine if content can be read or written to or
    * from the channel this socket can be used with a selector. This
    * provides a means to react to I/O events as they occur rather
    * than polling the channel which is generally less performant.
    *
    * @return this returns the connected socket channel
    */ 
   public SocketChannel getSocket();
 
   /**
    * This provides the <code>Cursor</code> for this channel. The
    * cursor provides a resettable view of the input buffer and will
    * allow the server kernel to peek into the input buffer without
    * having to take the data from the input. This allows overflow
    * to be pushed back on to the cursor for subsequent reads.
    *
    * @return this returns the input cursor for the channel
    */   
   public Cursor getCursor();
  
   /**
    * This provides the <code>Sender</code> for the channel. This is
    * used to provide a blocking output mechanism for the channel.
    * Enabling blocking reads ensures that output buffering can be
    * limited to an extent, which ensures that memory remains low at
    * high load periods. Writes to the sender may result in the data
    * being copied and queued until the socket is write ready.
    *
    * @return this returns the output sender for this channel
    */  
   public Sender getSender();
 
   /**
    * This returns the <code>Map</code> of attributes used to hold
    * connection information for the channel. The attributes here 
    * are taken from the pipeline attributes and may contain details
    * such as SSL certificates or other such useful information.
    *
    * @return returns the attributes associated with the channel
    */   
   public Map getAttributes();   
   
   /**
    * Because the channel represents a duplex means of communication
    * there needs to be a means to close it down. This provides such
    * a means. By closing the channel the cursor and sender will no
    * longer send or recieve data to or from the network. The client
    * will also be signaled that the connection has been severed.
    */  
   public void close();
   
}