/*
 * Container.java February 2001
 *
 * Copyright (C) 2001, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.http.core;

import org.simpleframework.http.Request;
import org.simpleframework.http.Response;

/**
 * The <code>Container</code> object is used to process HTTP requests
 * and compose HTTP responses. The <code>Request</code> objects that
 * are handed to this container contain all information relating to 
 * the received message. The responsibility of the container is to
 * interpret the request and compose a suitable response.
 * <p>
 * All implementations must ensure that the container is thread safe
 * as it will receive multiple HTTP transactions concurrently. Also
 * it should be known that the <code>Response</code> object used to
 * deliver the HTTP response will only commit and send once it has 
 * its <code>OutputStream</code> closed.
 * <p>
 * The <code>Container</code> is entirely responsible for the HTTP 
 * message headers and body. It is up to the implementation to ensure
 * that it complies to RFC 2616 or any previous specification. All 
 * headers and the status line can be modified by this object.
 * 
 * @author Niall Gallagher
 */
public interface Container {

   /**
    * Used to pass the <code>Request</code> and <code>Response</code>
    * to the container for processing. Any implementation of this 
    * must ensure that this is thread safe, as it will receive many
    * concurrent invocations each with a unique HTTP request.
    * <p>
    * The request and response objects are used to interact with the
    * connected pipeline, in such a way that requests and response
    * objects can be delivered in sequence and without interference.
    * The next request from a HTTP pipeline is only processed once
    * the <code>Response</code> object has been closed and committed.
    *
    * @param req the request that contains the client HTTP message
    * @param resp the response used to deliver the server response
    */
   public void handle(Request req, Response resp);
}
