/*
 * ContainerProcessor.java February 2007
 *
 * Copyright (C) 2007, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.http.core;

import java.io.IOException;

import org.simpleframework.transport.Processor;
import org.simpleframework.transport.Transport;
import org.simpleframework.util.buffer.Allocator;
import org.simpleframework.util.thread.Daemon;

/**
 * The <code>ContainerProcessor</code> object is used to create 
 * channels which can be used to consume and process requests. This
 * is basically an adapter to the <code>Selector</code> which will
 * convert the provided transport to a usable channel. Each of the
 * connected pipelines will end up at this object, regardless of
 * whether those connections are SSL or plain data.
 * 
 * @author Niall Gallagher
 */
public class ContainerProcessor implements Processor {
   
   /**
    * This is the allocator used internally be this processor.
    */
   private final Allocator allocator;
   
   /**
    * This is the selector used to process the created channels.
    */
   private final Selector selector;

   /**
    * Constructor for the <code>ContainerProcessor</code> object.
    * This is used to create a processor which will convert the
    * provided transport objects to channels, which can then be
    * processed by the selector and dispatched to the container.
    * 
    * @param container the container to dispatch requests to
    * @param allocator this is the allocator used to buffer data
    * @param count this is the number of threads to be used
    */
   public ContainerProcessor(Container container, Allocator allocator, int count) throws IOException {
     this(container, allocator, count, 1);
   }  
 
   /**
    * Constructor for the <code>ContainerProcessor</code> object.
    * This is used to create a processor which will convert the
    * provided transport objects to channels, which can then be
    * processed by the selector and dispatched to the container.
    * 
    * @param container the container to dispatch requests to
    * @param allocator this is the allocator used to buffer data
    * @param count this is the number of threads to be used
    * @param select this is the number of selector threads to use
    */
   public ContainerProcessor(Container container, Allocator allocator, int count, int select) throws IOException {
     this.selector = new ContainerSelector(container, allocator, count, select);
     this.allocator = allocator;
   }        

   /**
    * This is used to process the requests from a provided transport
    * and deliver a response to those requests. A transport can be
    * a direct transport or a secure transport providing SSL.  
    * <p>
    * Typical usage of this method is to accept multiple transport 
    * objects, each representing a unique HTTP channel to the client,
    * and process requests from those transports concurrently.  
    *      
    * @param transport the transport to process requests from
    */   
   public void process(Transport transport) throws IOException {
      selector.start(new TransportChannel(transport));
   }
   
   /**
    * This method is used to stop the <code>Processor</code> such 
    * that it will accept no more pipelines. Stopping the processor
    * ensures that all resources occupied will be released. This is 
    * required so that all threads are stopped and released.
    * <p>
    * Typically this method is called once all connections to the
    * server have been stopped. As a final act of shutting down the
    * entire server all threads must be stopped, this allows collection
    * of unused memory and the closing of file and socket resources.
    */    
   public void stop() throws IOException {
      selector.stop();
      allocator.close();
   }
 }