/*
 * Cursor.java February 2007
 *
 * Copyright (C) 2007, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.http.core;

import java.io.IOException;

/**
 * The <code>Cursor</code> object is used to acquire bytes from a 
 * given source. This provides a cursor style reading of bytes from 
 * a stream in that it will allow the reader to move the cursor back
 * if the amount of bytes read is too much. Allowing the cursor to 
 * move ensures that excess bytes back be placed back in the stream.
 * <p>
 * This is used when parsing input from a stream as it ensures that
 * on arrival at a terminal token any excess bytes can be placed 
 * back in to the stream. This allows data to be read efficiently
 * in large chunks from blocking streams such as sockets.
 *
 * @author Niall Gallagher
 */ 
interface Cursor {
 
   /**
    * Determines whether the cursor is still open. The cursor is
    * considered open if there are still bytes to read. If there is
    * still bytes buffered and the underlying transport is closed
    * then the cursor is still considered open. 
    * 
    * @return true if the read method does not return a -1 value
    */
   public boolean isOpen() throws IOException;
   
   /**
    * Determines whether the cursor is ready for reading. When the
    * cursor is ready then it guarantees that some amount of bytes
    * can be read from the underlying stream without blocking.
    *
    * @return true if some data can be read without blocking
    */         
   public boolean isReady() throws IOException;

   /**
    * Provides the number of bytes that can be read from the stream
    * without blocking. This is typically the number of buffered or
    * available bytes within the stream. When this reaches zero then
    * the cursor may perform a blocking read.
    *
    * @return the number of bytes that can be read without blocking
    */ 
   public int ready() throws IOException;

   /**
    * Reads a block of bytes from the underlying stream. This will
    * read up to the requested number of bytes from the underlying
    * stream. If there are no ready bytes on the stream this will
    * block, much like the <code>InputStream</code> a minus one is
    * returned if there are no more bytes to be read.
    *
    * @param data this is the array to read the bytes in to 
    *
    * @return this returns the number of bytes read from the stream 
    */ 
   public int read(byte[] data) throws IOException;

   /**
    * Reads a block of bytes from the underlying stream. This will
    * read up to the requested number of bytes from the underlying
    * stream. If there are no ready bytes on the stream this will
    * block, much like the <code>InputStream</code> a minus one is
    * returned if there are no more bytes to be read.
    *
    * @param data this is the array to read the bytes in to
    * @param off this is the offset to begin writing the bytes to
    * @param len this is the number of bytes that are requested 
    *
    * @return this returns the number of bytes read from the stream 
    */ 
   public int read(byte[] data, int off, int len) throws IOException;

   /**
    * Moves the cursor backward within the stream. This ensures 
    * that any bytes read from the last read can be pushed back
    * in to the stream so that they can be read again. This will
    * throw an exception if the reset can not be performed.
    *
    * @param len this is the number of bytes to reset back
    *
    * @return this is the number of bytes that have been reset
    */
   public int reset(int len) throws IOException;   
}
