/*
 * PartHeaderConsumer.java February 2007
 *
 * Copyright (C) 2007, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.http.core;

import java.io.IOException;

import org.simpleframework.util.buffer.Allocator;
import org.simpleframework.util.buffer.Buffer;

/**
 * The <code>PartHeaderConsumer</code> object is used to consume the
 * header for a multipart message. This performs a parse of the 
 * HTTP headers within the message up to the terminal carrige return
 * and line feed token. Once this had been read the contents of the
 * header are appended to a buffer so they can be read later.
 *
 * @author Niall Gallagher 
 */ 
class PartHeaderConsumer extends SegmentConsumer {
   
   /**
    * This is used to allocate the internal buffer for the header.
    */
   private Allocator allocator;
   
   /**
    * This is the internal buffer used to store the header.
    */ 
   private Buffer buffer;
   
   /**
    * Constructor for the <code>PartHeaderConsumer</code> object. An
    * allocator is required so that the header consumer can create a
    * buffer to store the contents of the consumed message. 
    *
    * @param allocator this is the allocator used to create a buffer
    */ 
   public PartHeaderConsumer(Allocator allocator) {
      this.allocator = allocator;
   }
   
   /**
    * This is used to process the header consumer once all of the 
    * headers have been read. This will simply parse all of the
    * headers and append the consumed bytes to the internal buffer.
    * Appending the bytes ensures that the whole upload can be
    * put back together as a single byte stream if required.
    */ 
   @Override
   protected void process() throws IOException {
      headers();
      append();
   }
   
   /**
    * This is used to allocate the internal buffer and append the
    * consumed bytes to the buffer. Once the  header is added to
    * the internal buffer this is finished and the next part of
    * the upload can be consumed.
    */ 
   private void append() throws IOException {      
      if(buffer == null) {
         buffer = allocator.allocate(count);
      }
      buffer.append(array, 0, count);
   }
}


