/*
 * TokenConsumer.java February 2007
 *
 * Copyright (C) 2001, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.http.core;

import java.io.IOException;

import org.simpleframework.util.buffer.Allocator;
import org.simpleframework.util.buffer.Buffer;

/**
 * The <code>TokenConsumer</code> object is used to consume a token
 * from the cursor. Once the token has been consumed the consumer
 * is finished and the contents of the consumed token is appended
 * to an allocated buffer so that it can be extracted.
 * 
 * @author Niall Gallagher
 */
class TokenConsumer extends ArrayConsumer {
   
   /**
    * This is used to allocate a buffer to append the contents.
    */
   private Allocator allocator;
   
   /**
    * This is used to append the contents of consumed token.
    */
   private Buffer buffer;
   
   /**
    * This is the token that is to be consumed from the cursor.
    */
   private byte[] token;
   
   /**
    * This tracks the number of bytes that are read from the cursor.
    */
   private int pos;
   
   /**
    * This tracks the number of bytes that are read from the token.
    */
   private int scan;
   
   /**
    * This is the length of the token that is to be consumed.
    */
   private int length;
   
   /**
    * The <code>TokenConsumer</code> object is used to read a token
    * from the cursor. This tracks the bytes read from the cursor, 
    * when it has fully read the token bytes correctly it will 
    * finish and append the consumed bytes to a buffer.
    * 
    * @param allocator the allocator used to create a buffer
    * @param token this is the token that is to be consumed
    */
   public TokenConsumer(Allocator allocator, byte[] token) {
      this.allocator = allocator;
      this.length = token.length;
      this.token = token;
   }
   
   /**
    * This is used to append the consumed bytes to a created buffer
    * so that it can be used when he is finished. This allows the
    * contents to be read from an input stream or as a string.
    */
   @Override
   protected void process() throws IOException {
      if(buffer == null) {
         buffer = allocator.allocate(length);
      }
      buffer.append(token);
   }
   
   /**
    * This is used to scan the token from the array. Once the bytes
    * have been read from the consumed bytes this will return the
    * number of bytes that need to be reset within the buffer. 
    *
    * @return this returns the number of bytes to be reset
    */
   @Override
   protected int scan() {
      int length = count;
      
      while(pos < count) {
         if(array[pos++] != token[scan++]) {
            scan = 0;
         } 
         if(scan == token.length) {
            done = true;
            count = pos;
            pos = 0;
            return length - count;
         }         
      }  
      return 0;
   }
}
