/*
 * SocketWriter.java February 2008
 *
 * Copyright (C) 2008, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.transport;

import java.io.IOException;
import java.nio.channels.SocketChannel;

/**
 * The <code>SocketWriter</code> object is used to coalesce the
 * packets to be written in to a minimum size. Also this will queue
 * the packets to be written in the order they are provided to that
 * if the contents of the packets can not be fully written they 
 * can be flushed in the correct order.
 * 
 * @author Niall Gallagher
 * 
 * @see org.simpleframework.transport.SegmentBuilder
 */
class SocketWriter implements Writer {
   
   /**
    * This is the manager used to build the segments to be written.
    */
   private SegmentBuilder builder;
   
   /**
    * The socket channel that the byte contents are written to.
    */
   private SocketChannel channel;
   
   /**
    * This is used to determine whether the socket is closed.
    */
   private boolean closed;
   
   /**
    * Constructor for the <code>SocketWriter</code> object. This
    * is used to wrap the socket in an object that will queue and
    * coalesce the packets written. It ensures that the packets
    * that are sent are of a minimum size for performance.
    * 
    * @param socket this is the pipeline instance this wraps
    */
   public SocketWriter(Socket socket) {
      this(socket, 20480);
   }
    
   /**
    * Constructor for the <code>SocketWriter</code> object. This
    * is used to wrap the socket in an object that will queue and
    * coalesce the packets written. It ensures that the packets
    * that are sent are of a minimum size for performance.
    * 
    * @param socket this is the pipeline instance this wraps
    * @param limit this is the threshold for asynchronous buffers
    */
   public SocketWriter(Socket socket, int limit) {
      this.builder = new SegmentBuilder(limit);
      this.channel = socket.getChannel();
   }

   /**
    * This provides the socket for the writer. Providing this 
    * enables a <code>Reactor</code> to be used to determine when
    * the writer is write ready and thus when the writer can
    * be flushed if it contains packets that have not been written.
    * 
    * @return this returns the socket associated with this
    */
   public synchronized SocketChannel getChannel() {
      return channel;
   }
   
   /**
    * This is used to determine if the writer should block or not.
    * A writer will block only if there are shared packets still
    * within the write queue. When all shared packets have either
    * been written or duplicated then the writer does not need to
    * block any waiting threads and they can be released.
    * 
    * @return true if any writing thread should be blocked
    */
   public synchronized boolean isBlocking() throws IOException {
      return builder.isReference();
   }

   /**
    * This is used to write the packets to the writer which will
    * be either written to the underlying socket or queued until
    * such time as the socket is write ready. This will return true
    * if the packet has been written to the underlying transport.
    * 
    * @param packet this is the packet that is the be written
    * 
    * @return true if the packet has been written to the transport
    */
   public synchronized boolean write(Packet packet) throws IOException {
      Segment segment = builder.build(packet); 
      
      if(segment == null) {
         return true;
      }
      return flush(); 
   }
   
   /**
    * This is used to send the packets to the socket. This attempts
    * to write the provided packet to the underlying socket, if 
    * all of the bytes are written the the packet is closed. This
    * will return the number of bytes that have been written.
    * 
    * @param segment this is the packet that is the be sent
    * 
    * @return the number of bytes written to the underlying socket
    */
   private synchronized int write(Segment segment) throws IOException {
      int size = segment.write(channel);      
      int left = segment.length();
      
      if(left == 0) {
         segment.close();
      }
      if(size < 0) {
         throw new TransportException("Socket is closed");
      }
      return size;
   }
   
   /**
    * This is used to flush all queued packets to the underlying
    * socket. If all of the queued packets have been fully written
    * then this returns true, otherwise this will return false.
    * 
    * @return true if all queued packets are flushed to the socket
    */   
   public synchronized boolean flush() throws IOException {
      int count = builder.length(); 
      
      while(count > 0) {
         Segment segment = builder.build(); 
         
         if(segment != null) { 
            int size = write(segment); 
            
            if(size == 0) { 
               break;
            }
            count -= size;
         }
      }
      if(count > 0) {
         builder.compact();
      }
      return count <= 0; 
   }

   /**
    * This is used to close the writer and the underlying socket.
    * If a close is performed on the writer then no more bytes 
    * can be read from or written to the writer and the client 
    * will receive a connection close on their side.
    */  
   public synchronized void close() throws IOException {
      if(!closed) {    
         closed = true;
         builder.close();
         channel.socket().shutdownOutput(); // send FIN ACK
         channel.close();
      }
   }
}
