/*
 * PoolFactory.java February 2009
 *
 * Copyright (C) 2009, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.util.thread;

import java.util.concurrent.ThreadFactory;

/**
 * The <code>PoolFactory</code> object is used to create a thread
 * factory that will instantiate named threads. This provides a
 * means to name the thread started using the type of task the
 * pool is required to execute.
 * 
 * @author Niall Gallagher
 */
class PoolFactory implements ThreadFactory {
   
   /**
    * This is the type of the task this pool will execute.
    */
   private final Class type;
   
   /**
    * Constructor for the <code>PoolFactory</code> object. This 
    * will provide a thread factory that names the threads based 
    * on the type of <code>Runnable</code> the pool executes. Each
    * of the threads is given a unique sequence number.
    * 
    * @param type this is the type of runnable this will execute
    */
   public PoolFactory(Class type) {
      this.type = type;
   }

   /**
    * This is used to create a new thread. The new thread will be
    * given the simple name of the <code>Runnable</code> class that 
    * it accepts. Each thread is also given a unique sequence.
    * 
    * @param task this is the worker that the thread pool uses
    *  
    * @return this returns the thread that is to be used by this 
    */
   public Thread newThread(Runnable task) {
      String prefix = type.getSimpleName();
      String name = ThreadNamer.getName(prefix);
      
      return new Thread(task, name);
   } 
}
