/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class UnlabelParameterVersionRequest extends SsmRequest implements
        ToCopyableBuilder<UnlabelParameterVersionRequest.Builder, UnlabelParameterVersionRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(UnlabelParameterVersionRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Long> PARAMETER_VERSION_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("ParameterVersion").getter(getter(UnlabelParameterVersionRequest::parameterVersion))
            .setter(setter(Builder::parameterVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParameterVersion").build()).build();

    private static final SdkField<List<String>> LABELS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Labels")
            .getter(getter(UnlabelParameterVersionRequest::labels))
            .setter(setter(Builder::labels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Labels").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            PARAMETER_VERSION_FIELD, LABELS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String name;

    private final Long parameterVersion;

    private final List<String> labels;

    private UnlabelParameterVersionRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.parameterVersion = builder.parameterVersion;
        this.labels = builder.labels;
    }

    /**
     * <p>
     * The name of the parameter from which you want to delete one or more labels.
     * </p>
     * <note>
     * <p>
     * You can't enter the Amazon Resource Name (ARN) for a parameter, only the parameter name itself.
     * </p>
     * </note>
     * 
     * @return The name of the parameter from which you want to delete one or more labels.</p> <note>
     *         <p>
     *         You can't enter the Amazon Resource Name (ARN) for a parameter, only the parameter name itself.
     *         </p>
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The specific version of the parameter which you want to delete one or more labels from. If it isn't present, the
     * call will fail.
     * </p>
     * 
     * @return The specific version of the parameter which you want to delete one or more labels from. If it isn't
     *         present, the call will fail.
     */
    public final Long parameterVersion() {
        return parameterVersion;
    }

    /**
     * For responses, this returns true if the service returned a value for the Labels property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasLabels() {
        return labels != null && !(labels instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more labels to delete from the specified parameter version.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLabels} method.
     * </p>
     * 
     * @return One or more labels to delete from the specified parameter version.
     */
    public final List<String> labels() {
        return labels;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(parameterVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasLabels() ? labels() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UnlabelParameterVersionRequest)) {
            return false;
        }
        UnlabelParameterVersionRequest other = (UnlabelParameterVersionRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(parameterVersion(), other.parameterVersion())
                && hasLabels() == other.hasLabels() && Objects.equals(labels(), other.labels());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UnlabelParameterVersionRequest").add("Name", name()).add("ParameterVersion", parameterVersion())
                .add("Labels", hasLabels() ? labels() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "ParameterVersion":
            return Optional.ofNullable(clazz.cast(parameterVersion()));
        case "Labels":
            return Optional.ofNullable(clazz.cast(labels()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("ParameterVersion", PARAMETER_VERSION_FIELD);
        map.put("Labels", LABELS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<UnlabelParameterVersionRequest, T> g) {
        return obj -> g.apply((UnlabelParameterVersionRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SsmRequest.Builder, SdkPojo, CopyableBuilder<Builder, UnlabelParameterVersionRequest> {
        /**
         * <p>
         * The name of the parameter from which you want to delete one or more labels.
         * </p>
         * <note>
         * <p>
         * You can't enter the Amazon Resource Name (ARN) for a parameter, only the parameter name itself.
         * </p>
         * </note>
         * 
         * @param name
         *        The name of the parameter from which you want to delete one or more labels.</p> <note>
         *        <p>
         *        You can't enter the Amazon Resource Name (ARN) for a parameter, only the parameter name itself.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The specific version of the parameter which you want to delete one or more labels from. If it isn't present,
         * the call will fail.
         * </p>
         * 
         * @param parameterVersion
         *        The specific version of the parameter which you want to delete one or more labels from. If it isn't
         *        present, the call will fail.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameterVersion(Long parameterVersion);

        /**
         * <p>
         * One or more labels to delete from the specified parameter version.
         * </p>
         * 
         * @param labels
         *        One or more labels to delete from the specified parameter version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder labels(Collection<String> labels);

        /**
         * <p>
         * One or more labels to delete from the specified parameter version.
         * </p>
         * 
         * @param labels
         *        One or more labels to delete from the specified parameter version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder labels(String... labels);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SsmRequest.BuilderImpl implements Builder {
        private String name;

        private Long parameterVersion;

        private List<String> labels = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(UnlabelParameterVersionRequest model) {
            super(model);
            name(model.name);
            parameterVersion(model.parameterVersion);
            labels(model.labels);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Long getParameterVersion() {
            return parameterVersion;
        }

        public final void setParameterVersion(Long parameterVersion) {
            this.parameterVersion = parameterVersion;
        }

        @Override
        public final Builder parameterVersion(Long parameterVersion) {
            this.parameterVersion = parameterVersion;
            return this;
        }

        public final Collection<String> getLabels() {
            if (labels instanceof SdkAutoConstructList) {
                return null;
            }
            return labels;
        }

        public final void setLabels(Collection<String> labels) {
            this.labels = ParameterLabelListCopier.copy(labels);
        }

        @Override
        public final Builder labels(Collection<String> labels) {
            this.labels = ParameterLabelListCopier.copy(labels);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder labels(String... labels) {
            labels(Arrays.asList(labels));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UnlabelParameterVersionRequest build() {
            return new UnlabelParameterVersionRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
