/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.timestreamquery;

import java.net.URI;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.config.AwsClientOption;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.endpointdiscovery.EndpointDiscoveryRefreshCache;
import software.amazon.awssdk.core.endpointdiscovery.EndpointDiscoveryRequest;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.identity.spi.AwsCredentialsIdentity;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.timestreamquery.internal.TimestreamQueryServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.timestreamquery.model.AccessDeniedException;
import software.amazon.awssdk.services.timestreamquery.model.CancelQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.CancelQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.ConflictException;
import software.amazon.awssdk.services.timestreamquery.model.CreateScheduledQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.CreateScheduledQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.DeleteScheduledQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.DeleteScheduledQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.DescribeAccountSettingsRequest;
import software.amazon.awssdk.services.timestreamquery.model.DescribeAccountSettingsResponse;
import software.amazon.awssdk.services.timestreamquery.model.DescribeEndpointsRequest;
import software.amazon.awssdk.services.timestreamquery.model.DescribeEndpointsResponse;
import software.amazon.awssdk.services.timestreamquery.model.DescribeScheduledQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.DescribeScheduledQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.ExecuteScheduledQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.ExecuteScheduledQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.InternalServerException;
import software.amazon.awssdk.services.timestreamquery.model.InvalidEndpointException;
import software.amazon.awssdk.services.timestreamquery.model.ListScheduledQueriesRequest;
import software.amazon.awssdk.services.timestreamquery.model.ListScheduledQueriesResponse;
import software.amazon.awssdk.services.timestreamquery.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.timestreamquery.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.timestreamquery.model.PrepareQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.PrepareQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.QueryExecutionException;
import software.amazon.awssdk.services.timestreamquery.model.QueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.QueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.ResourceNotFoundException;
import software.amazon.awssdk.services.timestreamquery.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.timestreamquery.model.TagResourceRequest;
import software.amazon.awssdk.services.timestreamquery.model.TagResourceResponse;
import software.amazon.awssdk.services.timestreamquery.model.ThrottlingException;
import software.amazon.awssdk.services.timestreamquery.model.TimestreamQueryException;
import software.amazon.awssdk.services.timestreamquery.model.UntagResourceRequest;
import software.amazon.awssdk.services.timestreamquery.model.UntagResourceResponse;
import software.amazon.awssdk.services.timestreamquery.model.UpdateAccountSettingsRequest;
import software.amazon.awssdk.services.timestreamquery.model.UpdateAccountSettingsResponse;
import software.amazon.awssdk.services.timestreamquery.model.UpdateScheduledQueryRequest;
import software.amazon.awssdk.services.timestreamquery.model.UpdateScheduledQueryResponse;
import software.amazon.awssdk.services.timestreamquery.model.ValidationException;
import software.amazon.awssdk.services.timestreamquery.transform.CancelQueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.CreateScheduledQueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.DeleteScheduledQueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.DescribeAccountSettingsRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.DescribeEndpointsRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.DescribeScheduledQueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.ExecuteScheduledQueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.ListScheduledQueriesRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.PrepareQueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.QueryRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.UpdateAccountSettingsRequestMarshaller;
import software.amazon.awssdk.services.timestreamquery.transform.UpdateScheduledQueryRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link TimestreamQueryClient}.
 *
 * @see TimestreamQueryClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTimestreamQueryClient implements TimestreamQueryClient {
    private static final Logger log = Logger.loggerFor(DefaultTimestreamQueryClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private EndpointDiscoveryRefreshCache endpointDiscoveryCache;

    protected DefaultTimestreamQueryClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        if (clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED)) {
            this.endpointDiscoveryCache = EndpointDiscoveryRefreshCache.create(TimestreamQueryEndpointDiscoveryCacheLoader
                    .create(this));
            if (clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden()) {
                log.warn(() -> "Endpoint discovery is enabled for this client, and an endpoint override was also specified. This will disable endpoint discovery for methods that require it, instead using the specified endpoint override. This may or may not be what you intended.");
            }
        }
    }

    /**
     * <p>
     * Cancels a query that has been issued. Cancellation is provided only if the query has not completed running before
     * the cancellation request was issued. Because cancellation is an idempotent operation, subsequent cancellation
     * requests will return a <code>CancellationMessage</code>, indicating that the query has already been canceled. See
     * <a href="https://docs.aws.amazon.com/timestream/latest/developerguide/code-samples.cancel-query.html">code
     * sample</a> for details.
     * </p>
     *
     * @param cancelQueryRequest
     * @return Result of the CancelQuery operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have the necessary permissions to access the account settings.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ThrottlingException
     *         The request was throttled due to excessive requests.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.CancelQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/CancelQuery" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CancelQueryResponse cancelQuery(CancelQueryRequest cancelQueryRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, InvalidEndpointException, AwsServiceException,
            SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = cancelQueryRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(cancelQueryRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelQueryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelQuery");

            return clientHandler.execute(new ClientExecutionParams<CancelQueryRequest, CancelQueryResponse>()
                    .withOperationName("CancelQuery").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withRequestConfiguration(clientConfiguration).withInput(cancelQueryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Create a scheduled query that will be run on your behalf at the configured schedule. Timestream assumes the
     * execution role provided as part of the <code>ScheduledQueryExecutionRoleArn</code> parameter to run the query.
     * You can use the <code>NotificationConfiguration</code> parameter to configure notification for your scheduled
     * query operations.
     * </p>
     *
     * @param createScheduledQueryRequest
     * @return Result of the CreateScheduledQuery operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have the necessary permissions to access the account settings.
     * @throws ConflictException
     *         Unable to poll results for a cancelled query.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ServiceQuotaExceededException
     *         You have exceeded the service quota.
     * @throws ThrottlingException
     *         The request was throttled due to excessive requests.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.CreateScheduledQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/CreateScheduledQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateScheduledQueryResponse createScheduledQuery(CreateScheduledQueryRequest createScheduledQueryRequest)
            throws AccessDeniedException, ConflictException, InternalServerException, ServiceQuotaExceededException,
            ThrottlingException, ValidationException, InvalidEndpointException, AwsServiceException, SdkClientException,
            TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateScheduledQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateScheduledQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = createScheduledQueryRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(createScheduledQueryRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createScheduledQueryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createScheduledQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateScheduledQuery");

            return clientHandler.execute(new ClientExecutionParams<CreateScheduledQueryRequest, CreateScheduledQueryResponse>()
                    .withOperationName("CreateScheduledQuery").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(createScheduledQueryRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateScheduledQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a given scheduled query. This is an irreversible operation.
     * </p>
     *
     * @param deleteScheduledQueryRequest
     * @return Result of the DeleteScheduledQuery operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have the necessary permissions to access the account settings.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ThrottlingException
     *         The request was throttled due to excessive requests.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.DeleteScheduledQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/DeleteScheduledQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteScheduledQueryResponse deleteScheduledQuery(DeleteScheduledQueryRequest deleteScheduledQueryRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, InvalidEndpointException, AwsServiceException, SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteScheduledQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteScheduledQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = deleteScheduledQueryRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(deleteScheduledQueryRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteScheduledQueryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteScheduledQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteScheduledQuery");

            return clientHandler.execute(new ClientExecutionParams<DeleteScheduledQueryRequest, DeleteScheduledQueryResponse>()
                    .withOperationName("DeleteScheduledQuery").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteScheduledQueryRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteScheduledQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the settings for your account that include the query pricing model and the configured maximum TCUs the
     * service can use for your query workload.
     * </p>
     * <p>
     * You're charged only for the duration of compute units used for your workloads.
     * </p>
     *
     * @param describeAccountSettingsRequest
     * @return Result of the DescribeAccountSettings operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have the necessary permissions to access the account settings.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ThrottlingException
     *         The request was throttled due to excessive requests.
     * @throws InvalidEndpointException
     *         The requested endpoint is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.DescribeAccountSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/DescribeAccountSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAccountSettingsResponse describeAccountSettings(DescribeAccountSettingsRequest describeAccountSettingsRequest)
            throws AccessDeniedException, InternalServerException, ThrottlingException, InvalidEndpointException,
            AwsServiceException, SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAccountSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAccountSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = describeAccountSettingsRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(describeAccountSettingsRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAccountSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAccountSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAccountSettings");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAccountSettingsRequest, DescribeAccountSettingsResponse>()
                            .withOperationName("DescribeAccountSettings").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                            .withInput(describeAccountSettingsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAccountSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * DescribeEndpoints returns a list of available endpoints to make Timestream API calls against. This API is
     * available through both Write and Query.
     * </p>
     * <p>
     * Because the Timestream SDKs are designed to transparently work with the service’s architecture, including the
     * management and mapping of the service endpoints, <i>it is not recommended that you use this API unless</i>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You are using <a href="https://docs.aws.amazon.com/timestream/latest/developerguide/VPCEndpoints">VPC endpoints
     * (Amazon Web Services PrivateLink) with Timestream </a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Your application uses a programming language that does not yet have SDK support
     * </p>
     * </li>
     * <li>
     * <p>
     * You require better control over the client-side implementation
     * </p>
     * </li>
     * </ul>
     * <p>
     * For detailed information on how and when to use and implement DescribeEndpoints, see <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/Using.API.html#Using-API.endpoint-discovery"
     * >The Endpoint Discovery Pattern</a>.
     * </p>
     *
     * @param describeEndpointsRequest
     * @return Result of the DescribeEndpoints operation returned by the service.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws ThrottlingException
     *         The request was throttled due to excessive requests.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.DescribeEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/DescribeEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEndpointsResponse describeEndpoints(DescribeEndpointsRequest describeEndpointsRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeEndpointsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeEndpointsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEndpoints");

            return clientHandler.execute(new ClientExecutionParams<DescribeEndpointsRequest, DescribeEndpointsResponse>()
                    .withOperationName("DescribeEndpoints").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeEndpointsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeEndpointsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides detailed information about a scheduled query.
     * </p>
     *
     * @param describeScheduledQueryRequest
     * @return Result of the DescribeScheduledQuery operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have the necessary permissions to access the account settings.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ThrottlingException
     *         The request was throttled due to excessive requests.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.DescribeScheduledQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/DescribeScheduledQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeScheduledQueryResponse describeScheduledQuery(DescribeScheduledQueryRequest describeScheduledQueryRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, InvalidEndpointException, AwsServiceException, SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeScheduledQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeScheduledQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = describeScheduledQueryRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(describeScheduledQueryRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeScheduledQueryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeScheduledQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeScheduledQuery");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeScheduledQueryRequest, DescribeScheduledQueryResponse>()
                            .withOperationName("DescribeScheduledQuery").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                            .withInput(describeScheduledQueryRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeScheduledQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * You can use this API to run a scheduled query manually.
     * </p>
     * <p>
     * If you enabled <code>QueryInsights</code>, this API also returns insights and metrics related to the query that
     * you executed as part of an Amazon SNS notification. <code>QueryInsights</code> helps with performance tuning of
     * your query. For more information about <code>QueryInsights</code>, see <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/using-query-insights.html">Using query
     * insights to optimize queries in Amazon Timestream</a>.
     * </p>
     *
     * @param executeScheduledQueryRequest
     * @return Result of the ExecuteScheduledQuery operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have the necessary permissions to access the account settings.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ThrottlingException
     *         The request was throttled due to excessive requests.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.ExecuteScheduledQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/ExecuteScheduledQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ExecuteScheduledQueryResponse executeScheduledQuery(ExecuteScheduledQueryRequest executeScheduledQueryRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, InvalidEndpointException, AwsServiceException, SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExecuteScheduledQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ExecuteScheduledQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = executeScheduledQueryRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(executeScheduledQueryRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(executeScheduledQueryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, executeScheduledQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExecuteScheduledQuery");

            return clientHandler.execute(new ClientExecutionParams<ExecuteScheduledQueryRequest, ExecuteScheduledQueryResponse>()
                    .withOperationName("ExecuteScheduledQuery").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(executeScheduledQueryRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ExecuteScheduledQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a list of all scheduled queries in the caller's Amazon account and Region. <code>ListScheduledQueries</code>
     * is eventually consistent.
     * </p>
     *
     * @param listScheduledQueriesRequest
     * @return Result of the ListScheduledQueries operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have the necessary permissions to access the account settings.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ThrottlingException
     *         The request was throttled due to excessive requests.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.ListScheduledQueries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/ListScheduledQueries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListScheduledQueriesResponse listScheduledQueries(ListScheduledQueriesRequest listScheduledQueriesRequest)
            throws AccessDeniedException, InternalServerException, ThrottlingException, ValidationException,
            InvalidEndpointException, AwsServiceException, SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListScheduledQueriesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListScheduledQueriesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = listScheduledQueriesRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(listScheduledQueriesRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listScheduledQueriesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listScheduledQueriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListScheduledQueries");

            return clientHandler.execute(new ClientExecutionParams<ListScheduledQueriesRequest, ListScheduledQueriesResponse>()
                    .withOperationName("ListScheduledQueries").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(listScheduledQueriesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListScheduledQueriesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List all tags on a Timestream query resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ThrottlingException
     *         The request was throttled due to excessive requests.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InvalidEndpointException,
            AwsServiceException, SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = listTagsForResourceRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(listTagsForResourceRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(listTagsForResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * A synchronous operation that allows you to submit a query with parameters to be stored by Timestream for later
     * running. Timestream only supports using this operation with <code>ValidateOnly</code> set to <code>true</code>.
     * </p>
     *
     * @param prepareQueryRequest
     * @return Result of the PrepareQuery operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have the necessary permissions to access the account settings.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ThrottlingException
     *         The request was throttled due to excessive requests.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.PrepareQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/PrepareQuery" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PrepareQueryResponse prepareQuery(PrepareQueryRequest prepareQueryRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, InvalidEndpointException, AwsServiceException,
            SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PrepareQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PrepareQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = prepareQueryRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(prepareQueryRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(prepareQueryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, prepareQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PrepareQuery");

            return clientHandler.execute(new ClientExecutionParams<PrepareQueryRequest, PrepareQueryResponse>()
                    .withOperationName("PrepareQuery").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(prepareQueryRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PrepareQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <code>Query</code> is a synchronous operation that enables you to run a query against your Amazon Timestream
     * data.
     * </p>
     * <p>
     * If you enabled <code>QueryInsights</code>, this API also returns insights and metrics related to the query that
     * you executed. <code>QueryInsights</code> helps with performance tuning of your query. For more information about
     * <code>QueryInsights</code>, see <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/using-query-insights.html">Using query
     * insights to optimize queries in Amazon Timestream</a>.
     * </p>
     * <note>
     * <p>
     * The maximum number of <code>Query</code> API requests you're allowed to make with <code>QueryInsights</code>
     * enabled is 1 query per second (QPS). If you exceed this query rate, it might result in throttling.
     * </p>
     * </note>
     * <p>
     * <code>Query</code> will time out after 60 seconds. You must update the default timeout in the SDK to support a
     * timeout of 60 seconds. See the <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/code-samples.run-query.html">code sample</a>
     * for details.
     * </p>
     * <p>
     * Your query request will fail in the following cases:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you submit a <code>Query</code> request with the same client token outside of the 5-minute idempotency window.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you submit a <code>Query</code> request with the same client token, but change other parameters, within the
     * 5-minute idempotency window.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the size of the row (including the query metadata) exceeds 1 MB, then the query will fail with the following
     * error message:
     * </p>
     * <p>
     * <code>Query aborted as max page response size has been exceeded by the output result row</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If the IAM principal of the query initiator and the result reader are not the same and/or the query initiator and
     * the result reader do not have the same query string in the query requests, the query will fail with an
     * <code>Invalid pagination token</code> error.
     * </p>
     * </li>
     * </ul>
     *
     * @param queryRequest
     * @return Result of the Query operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have the necessary permissions to access the account settings.
     * @throws ConflictException
     *         Unable to poll results for a cancelled query.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws QueryExecutionException
     *         Timestream was unable to run the query successfully.
     * @throws ThrottlingException
     *         The request was throttled due to excessive requests.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.Query
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/Query" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public QueryResponse query(QueryRequest queryRequest) throws AccessDeniedException, ConflictException,
            InternalServerException, QueryExecutionException, ThrottlingException, ValidationException, InvalidEndpointException,
            AwsServiceException, SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<QueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                QueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = queryRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(queryRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(queryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, queryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Query");

            return clientHandler.execute(new ClientExecutionParams<QueryRequest, QueryResponse>().withOperationName("Query")
                    .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withRequestConfiguration(clientConfiguration).withInput(queryRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new QueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associate a set of tags with a Timestream resource. You can then activate these user-defined tags so that they
     * appear on the Billing and Cost Management console for cost allocation tracking.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ServiceQuotaExceededException
     *         You have exceeded the service quota.
     * @throws ThrottlingException
     *         The request was throttled due to excessive requests.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ServiceQuotaExceededException, ThrottlingException, ValidationException, InvalidEndpointException,
            AwsServiceException, SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = tagResourceRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(tagResourceRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withRequestConfiguration(clientConfiguration).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the association of tags from a Timestream query resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws ThrottlingException
     *         The request was throttled due to excessive requests.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InvalidEndpointException
     *         The requested endpoint is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            ThrottlingException, ResourceNotFoundException, InvalidEndpointException, AwsServiceException, SdkClientException,
            TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = untagResourceRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(untagResourceRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(untagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Transitions your account to use TCUs for query pricing and modifies the maximum query compute units that you've
     * configured. If you reduce the value of <code>MaxQueryTCU</code> to a desired configuration, the new value can
     * take up to 24 hours to be effective.
     * </p>
     * <note>
     * <p>
     * After you've transitioned your account to use TCUs for query pricing, you can't transition to using bytes scanned
     * for query pricing.
     * </p>
     * </note>
     *
     * @param updateAccountSettingsRequest
     * @return Result of the UpdateAccountSettings operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have the necessary permissions to access the account settings.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ThrottlingException
     *         The request was throttled due to excessive requests.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.UpdateAccountSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/UpdateAccountSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateAccountSettingsResponse updateAccountSettings(UpdateAccountSettingsRequest updateAccountSettingsRequest)
            throws AccessDeniedException, InternalServerException, ThrottlingException, ValidationException,
            InvalidEndpointException, AwsServiceException, SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAccountSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateAccountSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = updateAccountSettingsRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(updateAccountSettingsRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAccountSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAccountSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAccountSettings");

            return clientHandler.execute(new ClientExecutionParams<UpdateAccountSettingsRequest, UpdateAccountSettingsResponse>()
                    .withOperationName("UpdateAccountSettings").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(updateAccountSettingsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateAccountSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update a scheduled query.
     * </p>
     *
     * @param updateScheduledQueryRequest
     * @return Result of the UpdateScheduledQuery operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have the necessary permissions to access the account settings.
     * @throws InternalServerException
     *         An internal server error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ThrottlingException
     *         The request was throttled due to excessive requests.
     * @throws ValidationException
     *         Invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamQueryClient.UpdateScheduledQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-query-2018-11-01/UpdateScheduledQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateScheduledQueryResponse updateScheduledQuery(UpdateScheduledQueryRequest updateScheduledQueryRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, InvalidEndpointException, AwsServiceException, SdkClientException, TimestreamQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateScheduledQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateScheduledQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = updateScheduledQueryRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(updateScheduledQueryRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateScheduledQueryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateScheduledQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateScheduledQuery");

            return clientHandler.execute(new ClientExecutionParams<UpdateScheduledQueryRequest, UpdateScheduledQueryResponse>()
                    .withOperationName("UpdateScheduledQuery").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(updateScheduledQueryRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateScheduledQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        TimestreamQueryServiceClientConfigurationBuilder serviceConfigBuilder = new TimestreamQueryServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TimestreamQueryException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEndpointException")
                                .exceptionBuilderSupplier(InvalidEndpointException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("QueryExecutionException")
                                .exceptionBuilderSupplier(QueryExecutionException::builder).httpStatusCode(400).build());
    }

    @Override
    public final TimestreamQueryServiceClientConfiguration serviceClientConfiguration() {
        return new TimestreamQueryServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
